<?php
// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class CSP_Admin_Settings {

    private $options;

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'add_plugin_page' ] );
        add_action( 'admin_init', [ $this, 'page_init' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_styles' ] );
    }

    public function enqueue_styles( $hook ) {
        // Only load on our plugin's settings page
        if ( 'toplevel_page_csp-settings' !== $hook ) {
            return;
        }
        wp_enqueue_style(
            'csp-admin-style',
            CSP_PLUGIN_URL . 'admin/css/admin-style.css',
            [],
            CSP_VERSION
        );
    }

    public function add_plugin_page() {
        add_menu_page(
            __( 'Comments Protection', 'comments-spam-protect' ),
            __( 'Comments Protection', 'comments-spam-protect' ),
            'manage_options',
            'csp-settings',
            [ $this, 'create_admin_page' ],
            'dashicons-shield-alt',
            80
        );
    }

    public function create_admin_page() {
        $this->options = get_option( 'csp_options' );
        ?>
        <div class="csp-wrap">
            <h1><?php esc_html_e( 'Comments Disable & Spam Protection', 'comments-spam-protect' ); ?></h1>
            <p><?php esc_html_e( 'Globally disable comments for any post type. This will override all individual post settings.', 'comments-spam-protect' ); ?></p>
            
            <form method="post" action="options.php">
                <?php
                settings_fields( 'csp_option_group' );
                do_settings_sections( 'csp-setting-admin' );
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public function page_init() {
        register_setting(
            'csp_option_group',
            'csp_options',
            [ $this, 'sanitize' ]
        );

        add_settings_section(
            'csp_setting_section',
            __( 'General Settings', 'comments-spam-protect' ),
            null,
            'csp-setting-admin'
        );

        add_settings_field(
            'disable_for_posts',
            __( 'Disable Comments on Posts', 'comments-spam-protect' ),
            [ $this, 'disable_for_posts_callback' ],
            'csp-setting-admin',
            'csp_setting_section'
        );

        add_settings_field(
            'disable_for_pages',
            __( 'Disable Comments on Pages', 'comments-spam-protect' ),
            [ $this, 'disable_for_pages_callback' ],
            'csp-setting-admin',
            'csp_setting_section'
        );

        add_settings_field(
            'disable_for_media',
            __( 'Disable Comments on Media / Attachments', 'comments-spam-protect' ),
            [ $this, 'disable_for_media_callback' ],
            'csp-setting-admin',
            'csp_setting_section'
        );
        
        if ( class_exists( 'WooCommerce' ) ) {
            add_settings_field(
                'disable_for_woocommerce',
                __( 'Disable Comments (Reviews) on WooCommerce Products', 'comments-spam-protect' ),
                [ $this, 'disable_for_woocommerce_callback' ],
                'csp-setting-admin',
                'csp_setting_section'
            );
        }
    }

    public function sanitize( $input ) {
        $sanitized_input = [];
        if ( isset( $input['disable_for_posts'] ) ) {
            $sanitized_input['disable_for_posts'] = 1;
        }
        if ( isset( $input['disable_for_pages'] ) ) {
            $sanitized_input['disable_for_pages'] = 1;
        }
        if ( isset( $input['disable_for_media'] ) ) {
            $sanitized_input['disable_for_media'] = 1;
        }
        if ( isset( $input['disable_for_woocommerce'] ) ) {
            $sanitized_input['disable_for_woocommerce'] = 1;
        }
        return $sanitized_input;
    }
    
    private function render_toggle_switch( $id, $option_name ) {
        $checked = isset( $this->options[$option_name] ) && $this->options[$option_name] == 1 ? 'checked' : '';
        printf(
            '<label class="csp-switch"><input type="checkbox" id="%s" name="csp_options[%s]" value="1" %s /><span class="slider round"></span></label>',
            esc_attr( $id ),
            esc_attr( $option_name ),
            $checked
        );
    }

    public function disable_for_posts_callback() {
        $this->render_toggle_switch('disable_for_posts', 'disable_for_posts');
    }

    public function disable_for_pages_callback() {
        $this->render_toggle_switch('disable_for_pages', 'disable_for_pages');
    }

    public function disable_for_media_callback() {
        $this->render_toggle_switch('disable_for_media', 'disable_for_media');
    }

    public function disable_for_woocommerce_callback() {
        $this->render_toggle_switch('disable_for_woocommerce', 'disable_for_woocommerce');
    }
}