<?php
// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class CSP_Core_Functions {

    private $options;
    private $post_types_to_disable = [];

    public function __construct() {
        $this->options = get_option( 'csp_options' );
        $this->build_post_types_list();

        if ( empty( $this->post_types_to_disable ) ) {
            return; // Nothing to do if no options are set
        }

        // Filters to close comments on the front-end
        add_filter( 'comments_open', [ $this, 'filter_comments_open' ], 20, 2 );
        add_filter( 'pings_open', [ $this, 'filter_comments_open' ], 20, 2 );

        // Hide existing comments
        add_filter( 'comments_array', [ $this, 'filter_comments_array' ], 20, 2 );

        // Admin area modifications
        add_action( 'admin_init', [ $this, 'remove_comment_support' ] );
        add_action( 'admin_menu', [ $this, 'remove_comments_menu_item' ] );
        add_action( 'wp_before_admin_bar_render', [ $this, 'remove_comments_from_admin_bar' ] );
    }

    private function build_post_types_list() {
        if ( ! empty( $this->options['disable_for_posts'] ) ) {
            $this->post_types_to_disable[] = 'post';
        }
        if ( ! empty( $this->options['disable_for_pages'] ) ) {
            $this->post_types_to_disable[] = 'page';
        }
        if ( ! empty( $this->options['disable_for_media'] ) ) {
            $this->post_types_to_disable[] = 'attachment';
        }
        if ( ! empty( $this->options['disable_for_woocommerce'] ) ) {
            $this->post_types_to_disable[] = 'product';
        }
    }

    public function filter_comments_open( $open, $post_id ) {
        $post = get_post( $post_id );
        if ( in_array( $post->post_type, $this->post_types_to_disable ) ) {
            return false;
        }
        return $open;
    }
    
    public function filter_comments_array( $comments, $post_id ) {
        if ( in_array( get_post_type( $post_id ), $this->post_types_to_disable ) ) {
            return []; // Return an empty array of comments
        }
        return $comments;
    }
    
    public function remove_comment_support() {
        foreach ( $this->post_types_to_disable as $post_type ) {
            // Removes the 'Comments' and 'Trackbacks' meta boxes from the post editor
            remove_post_type_support( $post_type, 'comments' );
            remove_post_type_support( $post_type, 'trackbacks' );
        }
    }

    public function remove_comments_menu_item() {
        // Hides the 'Comments' menu item in the admin dashboard
        remove_menu_page( 'edit-comments.php' );
    }
    
    public function remove_comments_from_admin_bar() {
        global $wp_admin_bar;
        $wp_admin_bar->remove_menu( 'comments' );
    }
}